/************************************************************************
 * FILE:   ajax.js                                                      *
 * AUTHOR: Justin Spargur                                               *
 * DATE:   04/21/09                                                     *
 *                                                                      *
 * DESCRIPTION:                                                         *
 * This file contains the JavaScript AJAX Object. This code is largely  *
 * based on "The Ultimate Ajax Object", found at                        *
 * http://www.hunlock.com/blogs/The_Ultimate_Ajax_Object                *
 * Modifications to this code were all made by 100 Pianos Web           *
 * Development and Design.                                              *
 *                                                                      *
 * GLOBAL VARIABLES:                                                    *
 *    - None                                                            *
 *                                                                      *
 * FUNCTIONS/METHODS INCLUDED:                                          *
 *    - Ajax           - This function is used to create an AJAX        *
 *                       object. This object can then be used to make   *
 *                       HTTP request (via GET or  POST) and process    *
 *                       the results.                                   *
 *                                                                      *
 *    - getParsedArray - Processes return values from an HTTP request   *
 *                       by splitting up individual lines and then      *
 *                       passing them to getParsedLine for further      *
 *                       parsing. Used by Ajax object for parsing       *
 *                       return values.                                 *
 *                                                                      *
 *    - getParsedLine  - Processes return values from an HTTP request   *
 *                       by parsing the data based on standard URL GET  *
 *                       parameter encoding                             *
 *                       (i.e., paramName1=paramValue1&paramName2...).  *
 *                       Used by Ajax object for parsing return values. *
 *                                                                      *
 * KNOWN BUGS:                                                          *
 *    - None                                                            *
 *                                                                      *
 * ADDITIONAL NOTES:                                                    *
 *    - None                                                            *
 *                                                                      *
 * CHANGES MADE:                                                        *
 *    - None                                                            *
 *                                                                      *
 ************************************************************************/




/************************************************************************
 *                                                                      *
 *  CLASS: Ajax                                                         *
 *                                                                      *
 *  PURPOSE:  Provide a JavaScript object for processing AJAX requests. *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 *  CONSTANTS:  None                                                    *
 *                                                                      *
 *  CLASS PARAMETER:                                                    *
 *     - URL              - URL to be used for the HTTP request.        *
 *                                                                      *
 *     - callbackFunction - Function that should be called once the     *
 *                          HTTP request has completed. The function    *
 *                          take the parameters, "response" and         *
 *                          "status" where "response" is the data being *
 *                          returned from the HTTP request and "status" *
 *                          is the HTTP status code being returned by   *
 *                          the request (200 = successful).             *
 *                                                                      *
 *  CLASS PROPERTIES:                                                   *
 *     - working       - Used to determine if the AJAX object is        *
 *                       processing a request. Either false or a        *
 *                       JavaScript Date object.                        *
 *                                                                      *
 *     - returnType    - Used to specify what format information should *
 *                       be returned in. Valid options are either       *
 *                       "array" or "text" (default).                   *
 *                                                                      *
 *     - parsedArray   - Used to store the return value from an AJAX    *
 *                       call in the form of an array.                  *
 *                                                                      *
 *  CLASS FUNCTIONS:                                                    *
 *     - send          - Used to send an HTTP request.                  *
 *                       PARAMETERS:                                    *
 *                          - passData - Data to be passed.  Should be  *
 *                              formated using standard URL GET         *
 *                              parameter encoding                      *
 *                              (i.e., paramName1=paramValue1...).      *
 *                                                                      *
 *                          - postMethod - HTTP protocol to use for the *
 *                              HTTP request. Either "POST" or "GET".   *
 *                                                                      *
 *                       RETURNS: A boolean true or false.              *
 *                                                                      *
 *                                                                      *
 *     - abort         - Used to cancel an HTTP request.                *
 *                       PARAMETERS: None                               *
 *                       RETURNS:    None                               *
 *                                                                      *
 ************************************************************************/
function Ajax(URL, callbackFunction)
{
   var that=this;
   this.working = false;
   this.returnType = "text";
   this.parsedArray = new Array();
   this.abort = function() {
      if (this.working)
      {
         that.working=false;
         that.AJAX.abort();
         that.AJAX=null;
      }
   }

   this.send = function(passData,postMethod)
   {
      if (that.working)
      {
         return false;
      }
      that.AJAX = null;
      if (window.XMLHttpRequest)
      {
         that.AJAX=new XMLHttpRequest();
      }
      else
      {
         that.AJAX=new ActiveXObject("Microsoft.XMLHTTP");
      }
      if (that.AJAX==null)
      {
         return false;
      }
      else
      {
         that.AJAX.onreadystatechange = function()
         {
            if (that.AJAX.readyState==4)
            {
               if(that.returnType == "array")
               {
                  // Parse array from results.
                  that.parsedArray = getParsedArray(that.AJAX.responseText);
                  that.callback(that.parsedArray,that.AJAX.status,that.AJAX.responseXML);
               }
               else
                  that.callback(that.AJAX.responseText,that.AJAX.status,that.AJAX.responseXML);
               that.working=false;
               that.AJAX=null;
            }
         }
         that.working = new Date();
         if (/post/i.test(postMethod))
         {
            var uri=urlCall+'?'+that.working.getTime();
            that.AJAX.open("POST", uri, true);
            that.AJAX.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
            that.AJAX.send(passData);
         }
         else
         {
            var uri=urlCall+'?'+passData+'&timestamp='+(that.working.getTime());
            that.AJAX.open("GET", uri, true);
            that.AJAX.send(null);
         }
         return true;
      }
   }

   var urlCall = URL;
   this.callback = callbackFunction || function () { };
}



/*********************************************************
 * FUNCTION: getParsedArray                              *
 *                                                       *
 * PURPOSE: Processes return values from an HTTP request *
 *          by splitting up individual lines and then    *
 *          passing them to getParsedLine for further    *
 *          parsing. Used by Ajax object for parsing     *
 *          return values.                               *
 *                                                       *
 * PARAMETERS:                                           *
 *     urlString - String data to be processed.          *
 *                                                       *
 * RETURNS: An array containing arrays representing      *
 *          input from the HTTP request return values.   *
 *                                                       *
 *********************************************************/
function getParsedArray(urlString)
{
   // Set up return variable.
   var returnArray = new Array();
   
   // Break up more than one line
   var lines = urlString.split("\n");

   for(var a = 0; a < lines.length; a++)
   {
      returnArray[a] = getParsedLine(lines[a]);
   }
   
   return returnArray;
}


/*********************************************************
 * FUNCTION: getParsedLine                               *
 *                                                       *
 * PURPOSE: Processes return values from an HTTP request *
 *          by parsing the data based on standard URL    *
 *          GET parameter encoding                       *
 *          (i.e., paramName1=paramValue1...).           *
 *                                                       *
 * PARAMETERS:                                           *
 *          line - String data to be processed.          *
 *                                                       *
 * RETURNS: An array representing input from the HTTP    *
 *          request return values.                       *
 *                                                       *
 *********************************************************/
function getParsedLine(line)
{
   var returnArray = new Array();
   
   var parsed = line.split("&");

   for(var a = 0; a < parsed.length; a++)
   {
      var tmp = parsed[a].split("=");
      if(tmp.length > 1)
      {
         returnArray[a] = tmp[1];
         returnArray[tmp[0]] = tmp[1];
      }
      else
         returnArray[a] = tmp[0];
   }

   return returnArray;
}