<?php
/************************************************************************
 * FILE:   class.pdf.php                                                *
 * AUTHOR: Justin Spargur                                               *
 * DATE:   04/21/09                                                     *
 *                                                                      *
 * INCLUDED FILES:                                                      *
 *    - "pdf/fpdf.php"                                                  *
 *                                                                      *
 * CLASS NAME: PDF                                                      *
 *                                                                      *
 *  PURPOSE:   Allows for the creation of PDF documents via PHP.        *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 *  EXTENDS:  FPDF class                                                *
 *                                                                      *
 *  ADDITIONAL CONSTANTS:  None                                         *
 *                                                                      *
 *  ADDITIONAL CLASS PROPERTIES:  None                                  *
 *                                                                      *
 *  ADDITIONAL CLASS FUNCTIONS:                                         *
 *     - Header            - Called every time a page is added. Creates *
 *                           a consistent header on every page.         *
 *                                                                      *
 *     - Footer            - Called just before a page is added.        *
 *                           Creates a consistent footer on every page. *
 *                                                                      *
 *     - DrawHR            - Creates a horizontal rule on the page.     *
 *                                                                      *
 *     - DrawFairnessScale - Creates a fairness scale on the page.      *
 *                                                                      *
 *     - putCheck          - Creates a checkbox at the current position *
 *                           on the fairness scale.                     *
 *                                                                      *
 * ADDITIONAL NOTES:                                                    *
 *    - For more documentation on the class, please see the parent      *]
 *      class documentation found at http://www.fpdf org/.              *
 *                                                                      *
 ************************************************************************/

require('fpdf.php');

class PDF extends FPDF
{
   /*********************************************************
    * FUNCTION: Header                                      *
    *                                                       *
    * PURPOSE: Called every time a page is added. Creates   *
    *          a consistent header on every page.           *
    *                                                       *
    * PARAMETERS: None                                      *
    *                                                       *
    * RETURNS: None                                         *
    *                                                       *
    *********************************************************/
   function Header()
   {
      //Logo
      //$tmp = $this->Image('asdmBannerMenu.jpg',$this->lMargin,$this->tMargin, $this->w - $this->lMargin - $this->rMargin);
      //$this->Ln($tmp["h"]);
   }

   /*********************************************************
    * FUNCTION: Footer                                      *
    *                                                       *
    * PURPOSE: Called just before a page is added. Creates  *
    *          a consistent footer on every page.           *
    *                                                       *
    * PARAMETERS: None                                      *
    *                                                       *
    * RETURNS: None                                         *
    *                                                       *
    *********************************************************/
   function Footer()
   {
      //Position at 1.5 cm from bottom
      $this->SetY( (-1 * $this->bMargin) );
      //Arial italic 8
      $this->SetFont('Arial','I',8);

      $this->Cell($this->w-$this->lMargin-$this->rMargin, $this->h/20,'Page '.$this->PageNo().'',0,0,'C');

      //Page number
      $this->Cell($this->w-$this->lMargin-$this->rMargin, $this->h/20,'Page '.$this->PageNo().'',0,0,'C');
   }

   /*********************************************************
    * FUNCTION: DrawHR                                      *
    *                                                       *
    * PURPOSE: Creates a horizontal rule on the page.       *
    *                                                       *
    * PARAMETERS: - lineHeight - width of the horizontal    *
    *                            rule being added.          *
    *                                                       *
    *             - addMargins - whether padding should be  *
    *                            added before and after the *
    *                            horizontal rule.           *
    *                                                       *
    * RETURNS: None                                         *
    *                                                       *
    *********************************************************/
   function DrawHR($lineHeight=5, $addMargins=true)
   {
      $previousLineWidth = $this->LineWidth;

      if($addMargins)
      {
         $this->Ln();
         $this->Ln();
         $this->Ln();
      }

      $this->SetLineWidth($lineHeight);
      $this->Rect($this->GetX(),$this->GetY(),($this->pageWidth - ($this->defaultMargin*1.5)),"F");

      if($addMargins)
      {
         $this->Ln();
         $this->Ln();
      }

      $this->SetLineWidth($previousLineWidth);
   }

   /*********************************************************
    * FUNCTION: DrawFairnessScale                           *
    *                                                       *
    * PURPOSE: Creates a horizontal rule on the page.       *
    *                                                       *
    * PARAMETERS: - rate - indicates where a checkbox       *
    *                      be placed. Default is 0, which   *
    *                      means no checkbox.               *
    *                                                       *
    * RETURNS: None                                         *
    *                                                       *
    *********************************************************/
   function DrawFairnessScale($rate = 0)
   {

      $this->Ln();

      $previousLineWidth = $this->LineWidth;

      $this->SetLineWidth(1);
      $printableWidth = $this->pageWidth - ($this->defaultMargin*1.5);

      $scaleWidth = ($printableWidth)*.75; // 3/4 of the page
      $scaleMargin = (($printableWidth)-$scaleWidth)/2;

      $leftSide = $this->GetX()+$scaleMargin;
      $top = $this->GetY();

      $this->Line($leftSide,$top,$leftSide+$scaleWidth,$top);

      $this->Line($leftSide, $top-3, $leftSide, $top+3);
      $this->Line($leftSide+($scaleWidth/2), $top-3, $leftSide+($scaleWidth/2), $top+3);
      $this->Line($leftSide+$scaleWidth, $top-3, $leftSide+$scaleWidth, $top+3);


      $this->SetY($this->GetY()-14);
      $this->SetX($this->GetX()+20);
      $this->Write(28,"Fair");
      $this->SetX($this->GetX()+390);
      $this->Write(28,"Unfair");

      $this->Ln();
      $this->SetX($leftSide);
      $this->SetDrawColor(56,93,138);
      $spaceBetweenCells = ($scaleWidth-(15*7))/6;

      $this->SetFillColor(0,0,102);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 7)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(51,51,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 6)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(51,102,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 5)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(102,153,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 4)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(153,204,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 3)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(204,236,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 2)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->SetFillColor(235,247,255);
      $this->Cell(15,15,'',1,0,'C',true);
      if($rate == 1)
         $this->putCheck();
      $this->SetX($this->GetX()+$spaceBetweenCells);

      $this->Ln();
      $this->Ln();
      $this->Ln();

      $this->SetLineWidth($previousLineWidth);
   }

   /*********************************************************
    * FUNCTION: putCheck                                    *
    *                                                       *
    * PURPOSE: Creates a checkbox at the current position   *
    *          on the fairness scale.                       *
    *                                                       *
    * PARAMETERS: None                                      *
    *                                                       *
    * RETURNS: None                                         *
    *                                                       *
    *********************************************************/
   function putCheck()
   {
      $setBackX = $this->GetX();
      $setBackY = $this->GetY();
      $this->SetX($this->GetX()-13);
      $this->Image("check.png",$this->GetX(),$this->GetY(),12,14);
      $this->SetX($setBackX);
   }
}
?>